#include <assert.h>
#include <stdio.h>
#include <string.h>
#include "bn.h"
#include "test.h"

struct
{
	const char *hex;
	int nlimbs;
	uint32_t limbs[4];
} small_num[] =
{
	{
		.hex = "0",
		.nlimbs = 1,
		.limbs = { 0 },
	},
	{
		.hex = "1",
		.nlimbs = 1,
		.limbs = { 1 },
	},
	{
		.hex = "ab",
		.nlimbs = 1,
		.limbs = { 0xab },
	},
	{
		.hex = "abc",
		.nlimbs = 1,
		.limbs = { 0xabc },
	},
	{
		.hex = "AbCd",
		.nlimbs = 1,
		.limbs = { 0xabcd },
	},
	{
		.hex = "12345678",
		.nlimbs = 1,
		.limbs = { 0x12345678 },
	},
	{
		.hex = "1234567890",
		.nlimbs = 2,
		.limbs = { 0x34567890, 0x12 },
	},
	{
		.hex = "aaaaBBBBccccDDDDeeeeFFFF",
		.nlimbs = 3,
		.limbs = { 0xeeeeffff, 0xccccdddd, 0xaaaabbbb },
	},
	{
		.hex = "12345678876543211234567887654321",
		.nlimbs = 4,
		.limbs = { 0x87654321, 0x12345678, 0x87654321, 0x12345678 },
	},
	{
		.hex = "a000000001111111122222222",
		.nlimbs = 4,
		.limbs = { 0x22222222, 0x11111111, 0x00000000, 0xa },
	},
	{
		.hex = 0,
		.nlimbs = 0,
		.limbs = { 0 },
	}
};

const char *const invalid[] =
{
	"",
	"1A!",
	"-27",
	"ghx",
	" ",
	" 38",
	0,
};

int main(void)
{
	int do_reg_check = 1;
	for (int i = 0; small_num[i].hex; ++i)
	{
		bn *x = bn_new();
		REQUIRE(x, "x = bn_new() returned NULL");
		check_regs = do_reg_check;
		int ret = bn_hex2bn(x, small_num[i].hex);
		do_reg_check = check_regs = 0;
		TEST(!ret, "bn_hex2bn did not return 0");
		TEST(x->nlimbs == small_num[i].nlimbs, "nlimbs is wrong");
		TEST(x->nlimbs <= x->size, "size is smaller than nlimbs");
		TEST(!memcmp(x->d, small_num[i].limbs, 4*small_num[i].nlimbs),
		     "limbs are wrong");
		bn_free(x);
	}
	bn *x = bn_new();
	REQUIRE(x, "x = bn_new() returned NULL");
	for (int i = 0; invalid[i]; ++i)
		TEST(-1 == bn_hex2bn(x, invalid[i]), "did not reject invalid string");

	char big[513];
	memset(big, 'A', sizeof big - 1);
	big[512] = 0;
	REQUIRE(x, "x = bn_new() returned NULL");
	REQUIRE(!bn_hex2bn(x, "0"), "failed to set 0");

	int ret = bn_hex2bn(x, big);
	TEST(!ret, "bn_hex2bn did not return 0");
	TEST(x->nlimbs == 512/2/4, "nlimbs is wrong");
	TEST(x->nlimbs <= x->size, "size is smaller than nlimbs");
	uint32_t limbs[512/2/4];
	memset(limbs, 0xaa, sizeof limbs);
	TEST(!memcmp(x->d, limbs, sizeof limbs), "limbs are wrong");

	REQUIRE(-1 == bn_hex2bn(x, "invalid"), "did not reject invalid string");
	TEST(x->nlimbs == 512/2/4, "nlimbs is wrong");
	TEST(x->nlimbs <= x->size, "size is smaller than nlimbs");
	TEST(!memcmp(x->d, limbs, sizeof limbs), "limbs are wrong");

	bn_free(x);
	print_test_results(8);
	return failed_tests;
}
