#lang errortrace racket
(require racket/match)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; The heap data-structure
(provide
  (struct-out eff)
  handle?
  heap?
  heapof?
  heap-put
  heap-get
  handle
  effof?
  parse-handle
  heap-filter)

(struct handle (id) #:transparent #:guard
  (lambda (id name)
    (unless (exact-nonnegative-integer? id)
      (error "handle: id: expecting non-negative integer, got:" id))
    id))

(struct heap (data) #:transparent)

(define (heapof? value?)
  (struct/c heap (hash/c handle? value? #:immutable #t)))

(struct eff (state result) #:transparent)
(define (effof? state? result?)
  (struct/c eff state? result?))

(define empty-heap (heap (hash)))
(define/contract (heap-alloc h v)
  (-> (heapof? any/c) any/c eff?)
  (define data (heap-data h))
  (define new-id (handle (hash-count data)))
  (define new-heap (heap (hash-set data new-id v)))
  (eff new-heap new-id))
(define/contract (heap-get h k)
  (-> (heapof? any/c) handle? any/c)
  (hash-ref (heap-data h) k))
(define/contract (heap-put h k v)
  (-> (heapof? any/c) handle? any/c heap?)
  (define data (heap-data h))
  (cond
    [(hash-has-key? data k) (heap (hash-set data k v))]
    [else (error "Unknown handle!")]))
(define (nonempty-heapof? value?)
  (and/c (heapof? value?)
    (flat-named-contract 'nonempty
      (lambda (x) (> (hash-count (heap-data x)) 0)))))

(define (heap-fold proc init hp)
  (->
    ; (accum key val) -> accum
    (-> any/c handle? any/c any/c)
    any/c ; accum
    heap?)
  (foldl
    (lambda (accum elem) (proc accum (car elem) (cdr elem)))
    init
    (hash->list (heap-data hp))))

(define/contract (heap-filter proc hp)
  (->
    ; for each key val returns a boolean
    (-> handle? any/c boolean?)
    ; Given a heap
    heap?
    ; Returns a heap
    heap?)
  (heap
    (make-immutable-hash
      (filter
        (lambda (pair) (proc (car pair) (cdr pair)))
        (hash->list (heap-data hp))))))

(module+ test
  (require rackunit)
  (test-case
    "Simple"
    (define h1 empty-heap)          ; h is an empty heap
    (define r (heap-alloc h1 "foo")) ; stores "foo" in a new memory cell
    (define h2 (eff-state r))
    (define x (eff-result r)) ;
    (check-equal? "foo" (heap-get h2 x)) ; checks that "foo" is in x
    (define h3 (heap-put h2 x "bar"))    ; stores "bar" in x
    (check-equal? "bar" (heap-get h3 x)))) ; checks that "bar" is in x

(module+ test
  (test-case
    "Unique"
    (define h1 empty-heap)          ; h is an empty heap
    (define r1 (heap-alloc h1 "foo")) ; stores "foo" in a new memory cell
    (define h2 (eff-state r1))
    (define x (eff-result r1))
    (define r2 (heap-alloc h2 "bar")) ; stores "foo" in a new memory cell
    (define h3 (eff-state r2))
    (define y (eff-result r2))
    (check-not-equal? x y)  ; Ensures that x != y
    (check-equal? "foo" (heap-get h3 x))
    (check-equal? "bar" (heap-get h3 y))))

(module+ test
  ((thunk
    (define m6 (parse-mem '((E0 (a . 10) (x . 0)) (E1 E0 (b . 20) (x . 1)) (E2 E0 (a . 30)) (E3 E2 (z . 3)))))
    (check-equal?
      (heap-filter (lambda (r frm) (even? (handle-id r))) m6)
      (parse-mem '((E0 (a . 10) (x . 0)) (E2 E0 (a . 30))))))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(provide
  (struct-out s:void)
  (except-out (struct-out s:apply)
    s:apply-func s:apply-args)
  (except-out (struct-out s:lambda)
    s:lambda-params s:lambda-body)
  (except-out (struct-out s:number)
    s:number-value)
  (except-out (struct-out s:closure)
    s:closure-decl s:closure-env)
  (except-out (struct-out s:variable)
    s:variable-name)
  (except-out (struct-out s:define)
    s:define-var s:define-body)
  (except-out (struct-out s:seq)
    s:seq-fst s:seq-snd)
  (except-out (struct-out s:bool)
    s:bool-value)
  (except-out (struct-out s:builtin)
    s:builtin-func)
  s:value?
  s:expression?
  s:term?)

;; Values
(define (s:value? v)
  (or (s:number? v)
      (s:void? v)
      (s:closure? v)
      (s:bool? v)
      (s:builtin? v)))
(struct s:void () #:transparent)
(struct s:number (value) #:transparent)
(struct s:closure (env decl) #:transparent)
(struct s:bool (value) #:transparent)
(struct s:builtin (func) #:transparent)
;; Expressions
(define (s:expression? e)
  (or (s:value? e)
      (s:variable? e)
      (s:apply? e)
      (s:lambda? e)))
(struct s:lambda (params body) #:transparent)
(struct s:variable (name) #:transparent)
(struct s:apply (func args) #:transparent)
;; Terms
(define (s:term? t)
  (or (s:expression? t)
      (s:define? t)
      (s:seq? t)))
(struct s:define (var body) #:transparent)
(struct s:seq (fst snd) #:transparent)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Serializing an AST into a datum
(provide s:quote1 s:quote)

;; Quotes a sequence of terms
(define/contract (s:quote term)
  (-> s:term? list?)
  (match term
    [(s:seq t1 t2) (append (s:quote t1) (s:quote t2))]
    [_ (list (s:quote1 term))]))

(define/contract (quote-handle h)
  (-> handle? symbol?)
  (string->symbol (format "E~a" (handle-id h))))

;; Quote one term
(define/contract (s:quote1 term)
  (-> (or/c s:term? handle?) any/c)
  (match term
    [(s:lambda xs t)
     (cons 'lambda (cons (map s:quote1 xs) (s:quote t)))]
    [(s:apply ef ea)
     (cons (s:quote1 ef) (map s:quote1 ea))]
    [(s:number n) n]
    [(s:bool b) b]
    [(s:builtin f) (error (format "s:builtin cannot be quoted! Got: ~a" f))]
    [(s:variable x) x]
    [(s:define x (s:lambda xs t))
     (cons 'define (cons (map s:quote1 (cons x xs)) (s:quote t)))]
    [(s:define x e) (list 'define (s:quote1 x) (s:quote1 e))]
    [(s:closure env decl) (list 'closure (quote-handle env) (s:quote1 decl))]
    [(s:void) (list 'void)]
    [(? handle? x) (quote-handle x)]))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Parsing a datum into an AST
(provide s:parse1 s:parse)

(define (lambda? node)
  (and
    (list? node)
    (>= (length node) 3)
    (equal? 'lambda (first node))
    (list? (lambda-params node))
    (andmap symbol? (lambda-params node))))
(define lambda-params cadr)
(define lambda-body cddr)

(define (apply? l)
  (and (list? l) (>= (length l) 1)))
(define apply-func car)
(define apply-args cdr)

(define (define-basic? node)
  (and
    (list? node)
    (= (length node) 3)
    (equal? 'define (car node))
    (symbol? (define-head node))))

(define (define-func? node)
  (and
    (list? node)
    (>= (length node) 3)
    (equal? 'define (car node))
    (list? (define-head node))
    (andmap symbol? (define-head node))
    (>= (length (define-head node)) 1)))

(define (define? node)
  (or
    (define-basic? node)
    (define-func? node)))

(define define-head cadr)
(define define-body cddr)

(define (void? node)
  (and
    (list? node)
    (= (length node) 1)
    (equal? 'void (first node))))

(define (closure? node)
  ; (closure env decl)
  (and
    (list? node)
    (= (length node) 3)
    (equal? 'closure (first node))))

(define (s:parse node)
  (define (on-elem datum accum)
    (define elem (s:parse1 datum))
    (cond [(null? accum) elem]
          [else (s:seq elem accum)]))
  (define result (foldr on-elem null node))
  (when (null? result)
    (error "A list with 1 or more terms, but got:" node))
  result)

(define/contract (parse-handle node)
  (-> symbol? handle?)
  (handle (string->number (substring (symbol->string node) 1))))

(define (s:parse1 node)
  (define (build-lambda args body)
    (define (on-elem datum accum)
      (define elem (s:parse1 datum))
      (cond [(s:void? accum) elem]
            [else (s:seq elem accum)]))
    (s:lambda (map s:variable args) (foldr on-elem (s:void) body)))

  (define (make-define-func node)
    (s:define
      (s:variable (first (define-head node)))
      (build-lambda (rest (define-head node)) (define-body node))))

  (define (make-define-expr node)
    (s:define
      (s:variable (define-head node))
      (s:parse1 (first (define-body node)))))

  (define (make-lambda node)
    (build-lambda (lambda-params node) (lambda-body node)))

  (define (make-apply node)
    (s:apply (s:parse1 (first node)) (map s:parse1 (rest node))))

  (define (make-closure node)
    (s:closure (parse-handle (second node)) (s:parse1 (third node))))

  (cond
    [(define-basic? node) (make-define-expr node)]
    [(define-func? node) (make-define-func node)]
    [(symbol? node) (s:variable node)]
    [(real? node) (s:number node)]
    [(lambda? node) (make-lambda node)]
    [(closure? node) (make-closure node)]
    [(boolean? node) (s:bool node)]
    [(void? node) (s:void)]
    [else (make-apply node)]))

(define (quote-hash map quote-key quote-val lt?)
  (define (for-each k v)
    (cons (quote-key k) (quote-val v)))
  (define (<? x y)
    (lt? (car x) (car y)))
  (sort (hash-map map for-each) <?))

(define (parse-hash node parse-key parse-val)
  (define (for-each pair)
    (cons (parse-key (car pair)) (parse-val (cdr pair))))
  (make-immutable-hash (map for-each node)))

(module+ test
  (require rackunit)
  (define (check-parses1? exp)
    (check-equal? (s:quote1 (s:parse1 exp)) exp))
  (define (check-parses? term)
    (check-equal? (s:quote (s:parse term)) term))

  (check-equal? (s:parse1 '(lambda () x y)) (s:lambda (list) (s:seq (s:variable 'x) (s:variable 'y))))
  (check-equal? (s:parse1 '(lambda () x)) (s:lambda (list) (s:variable 'x)))
  (check-equal? (s:parse1 '(lambda () z y x)) (s:lambda (list) (s:seq (s:variable 'z) (s:seq (s:variable 'y) (s:variable 'x)))))
  (check-equal? (s:quote1 (s:lambda (list) (s:variable 'x))) '(lambda () x))
  (check-parses1? '(lambda () x y))
  (check-parses1? '(lambda () x))
  (check-parses1? '(lambda () x y z))
  (check-parses? '((define x 10) (define (f x) 10) (lambda () x y z)))
  (check-parses1? '(closure E1 (lambda (x) x)))
  (check-parses1? '#t)
  (check-parses1? '#f))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Frames
(provide
  parse-frame
  quote-frame
  frame-put
  root-frame
  frame-get
  frame-push
  frame-fold
  (struct-out frame))

(struct frame (parent locals) #:transparent)

;; Creates a curried binary-builtin
(define (binop f)
  ;; Returns a curried binary builtin
  (s:builtin
    (lambda (v1)
      (s:builtin
        (lambda (v2)
          (s:number (f (s:number-value v1) (s:number-value v2))))))))

(define root-empty-frame (frame #f (hash)))
(define root-frame (frame #f (hash (s:variable '+) (binop +) (s:variable '*) (binop *))))
(define (frame-push parent var val)
  (frame parent (hash var val)))
(define/contract (frame-put frm var val)
  (-> frame? s:variable? s:value? frame?)
  (frame (frame-parent frm) (hash-set (frame-locals frm) var val)))
(define/contract (frame-get frm var)
  (-> frame? s:variable? (or/c s:value? #f))
  (hash-ref (frame-locals frm) var #f))
(define/contract (frame-fold proc init frm)
  (-> (-> s:variable? s:value? any/c any/c) any/c frame? any/c)
  (foldl (lambda (pair accum) (proc (car pair) (cdr pair) accum)) init (hash->list (frame-locals frm))))
(define/contract (frame-values frm)
  (-> frame? (listof s:value?))
  (map cdr (hash->list (frame-locals frm))))

(define/contract (quote-frame frm)
  (-> frame? list?)
  (define hdl (cond [(frame-parent frm) (quote-handle (frame-parent frm))] [else #f]))
  (define elems (quote-hash (frame-locals frm) s:quote1 s:quote1 symbol<?))
  (if hdl (cons hdl elems) elems))

(define/contract (parse-frame node)
  (-> list? frame?)
  (define (on-handle node)
    (cond [(boolean? node) node]
          [else (parse-handle node)]))
  (define hd (if (or (empty? node) (pair? (first node))) #f (first node)))
  (define elems (if hd (rest node) node))
  (frame (on-handle hd) (parse-hash elems s:parse1 s:parse1)))

(module+ test
  (require rackunit)
  (define (check-parses-frame? frm)
    (define parsed (parse-frame frm))
    (define given (quote-frame parsed))
    (check-equal? given frm)
    (check-equal? (parse-frame given) parsed))
  (check-parses-frame? '(E1))
  (check-parses-frame? '())
  (check-parses-frame? '([x . 3] [y . 2]))
  (check-parses-frame? '(E2 [x . 3] [y . 2]))
  ;; Slide 1
  ; (closure E0 (lambda (y) a)
  (define c (s:closure (handle 0) (s:lambda (list (s:variable 'y)) (s:variable 'a))))
  ;E0: [
  ;  (a . 20)
  ;  (b . (closure E0 (lambda (y) a)))
  ;]
  (define f1
    (frame-put
      (frame-put root-empty-frame (s:variable 'a) (s:number 10))
      (s:variable 'b) c))
  (check-equal? f1 (frame #f (hash (s:variable 'a) (s:number 10) (s:variable 'b) c)))
  ; Lookup a
  (check-equal? (s:number 10) (frame-get f1 (s:variable 'a)))
  ; Lookup b
  (check-equal? c (frame-get f1 (s:variable 'b)))
  ; Lookup c that does not exist
  (check-equal? #f (frame-get f1 (s:variable 'c)))
  ;; Slide 2
  (define f2 (frame-push (handle 0) (s:variable 'y) (s:number 1)))
  (check-equal? f2 (frame (handle 0) (hash (s:variable 'y) (s:number 1))))
  (check-equal? (s:number 1) (frame-get f2 (s:variable 'y)))
  (check-equal? #f (frame-get f2 (s:variable 'a)))
  ;; We can use frame-parse to build frames
  (check-equal? (parse-frame '[ (a . 10) (b . (closure E0 (lambda (y) a)))]) f1)
  (check-equal? (parse-frame '[ E0 (y . 1) ]) f2))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Environment
(provide
  root-environ
  environ-push
  environ-put
  mem?
  environ-get
  s:eval-eff?
  root-mem
  parse-mem
  quote-mem)

;; The root environment is initialized with a single frame
(define mem? (nonempty-heapof? frame?))
(define root-alloc (heap-alloc empty-heap root-frame))
(define/contract root-environ handle? (eff-result root-alloc))
(define/contract root-mem mem? (eff-state root-alloc))
(define root-empty-mem (eff-state (heap-alloc empty-heap root-empty-frame)))

(define s:eval-eff? (effof? mem? s:value?))

;; The put operation
(define/contract (environ-put mem env var val)
  (-> mem? handle? s:variable? s:value? heap?)
  (define new-frm (frame-put (heap-get mem env) var val))
  (heap-put mem env new-frm))
;; The push operation
(define/contract (environ-push mem env var val)
  (-> mem? handle? s:variable? s:value? eff?)
  (define new-frame (frame env (hash var val)))
  (heap-alloc mem new-frame))
;; The Get operation
(define/contract (environ-get mem env var)
  (-> mem? handle? s:variable? s:value?)
  (define (environ-get-aux env)
    (define frm (heap-get mem env))    ;; Load the current frame
    (define parent (frame-parent frm))  ;; Load the parent
    (define result (frame-get frm var)) ;; Lookup locally
    (cond
      [result result] ;; Result is defined, then return it
      [parent (environ-get-aux parent)] ; If parent exists, recurse
      [else #f]))
  (define res (environ-get-aux env))
  ; Slight change from the slides for better error reporting
  (when (not res)
    (error
      (format "Variable ~a was NOT found in environment ~a. Memory dump:\n~a"
        (s:quote1 var)
        (quote-handle env)
        (quote-mem mem))))
  res)

(define/contract (parse-mem node)
  (-> any/c heap?)
  (heap (parse-hash node parse-handle parse-frame)))

(define (quote-mem mem)
  (-> heap? list?)
  (quote-hash (heap-data mem) quote-handle quote-frame symbol<?))

(module+ test
  (define (check-parses-mem? mem)
    (check-equal? (quote-mem (parse-mem mem)) mem))
  (check-parses-mem? '([E0 . ([x . 3] [y . 2])]))
  (define E0 root-environ)
  (define m1
    (environ-put
      (environ-put root-empty-mem E0 (s:variable 'x) (s:number 3))
      E0 (s:variable 'y) (s:number 5)))
  (define e1-m2 (environ-push m1 E0 (s:variable 'z) (s:number 6)))
  (define E1 (eff-result e1-m2))
  (define m2 (eff-state e1-m2))
  (define m3 (environ-put m2 E1 (s:variable 'x) (s:number 7)))
  (define e2-m4 (environ-push m3 E0 (s:variable 'm) (s:number 1)))
  (define E2 (eff-result e2-m4))
  (define m4 (eff-state e2-m4))
  (define m5 (environ-put m4 E2 (s:variable 'y) (s:number 2)))

  (define parsed-m5
    (parse-mem
      '([E0 . ([x . 3] [y . 5])]
        [E1 . (E0 [x . 7] [z . 6])]
        [E2 . (E0 [m . 1] [y . 2])])))
  (check-equal? parsed-m5 m5)
  (check-equal? (environ-get m5 E0 (s:variable 'x)) (s:number 3))
  (check-equal? (environ-get m5 E0 (s:variable 'y)) (s:number 5))
  (check-equal? (environ-get m5 E1 (s:variable 'x)) (s:number 7))
  (check-equal? (environ-get m5 E1 (s:variable 'z)) (s:number 6))
  (check-equal? (environ-get m5 E1 (s:variable 'y)) (s:number 5))
  (check-equal? (environ-get m5 E2 (s:variable 'y)) (s:number 2))
  (check-equal? (environ-get m5 E2 (s:variable 'm)) (s:number 1))
  (check-equal? (environ-get m5 E2 (s:variable 'x)) (s:number 3))
  (define m6 (parse-mem '((E0 (a . 10) (x . 0)) (E1 E0 (b . 20) (x . 1)) (E2 E0 (a . 30)) (E3 E2 (z . 3)))))
  (check-equal? (environ-get m6 E1 (s:variable 'b)) (s:number 20)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(provide
  (struct-out eff-op)
  eff-bind
  eff-pure
  eff-run
  do)

(struct eff-op (func))

(define (eff-run op h)
  ((eff-op-func op) h))

(define (eff-bind o1 o2)
  (eff-op
    (lambda (h1)
      (define h2+r (eff-run o1 h1))
      (define r (eff-result h2+r))
      (define h2 (eff-state h2+r))
      (eff-run (o2 r) h2))))

(define (eff-pure x)
  (eff-op (lambda (h) (eff h x))))

(define-syntax do
  (syntax-rules (<-)
    ; Only one monadic-op, return it
    [(_ mexp) mexp]
    ; A binding operation
    [(_ var <- mexp rest ...) (eff-bind mexp (lambda (var) (do rest ...)))]
    ; No binding operator, just ignore the return value
    [(_ mexp rest ...)        (eff-bind mexp (lambda (_) (do rest ...)))]))
